﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Api.Utilities;
    using Hims.Domain.Repositories.UnitOfWork;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.Common;
    using Microsoft.AspNetCore.Authorization;
    using Npgsql;
    using Shared.UserModels.Filters;

    /// <inheritdoc />
    [Authorize]
    [Route("api/scan-appointment-notice")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ScanAppointmentNoticeController : BaseController
    {
        public readonly IScanAppointmentNoticeService scanAppointmentNoticeService;

        /// <summary>
        /// The scan log services.
        /// </summary>
        private readonly IScanLogService scanLogServices;

        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        public ScanAppointmentNoticeController(IScanAppointmentNoticeService scanAppointmentNoticeService, IScanLogService scanLogServices, IUnitOfWork unitOfWork)
        {
            this.scanAppointmentNoticeService = scanAppointmentNoticeService;
            this.scanLogServices = scanLogServices;
            this.unitOfWork = unitOfWork;
        }

        [AllowAnonymous]
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ScanAppointmentNoticeModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ScanAppointmentNoticeModel model)
        {
            model = (ScanAppointmentNoticeModel)EmptyFilter.Handler(model);
            var response = await this.scanAppointmentNoticeService.FetchAsync(model);
            return response == null ? this.ServerError() : this.Success(response);
        }

        /// <summary>
        /// The delete  scan Appointment Notice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 -  scan Appointment Notice deleted successfully.
        /// - 409 - scan Appointment Notice can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ScanAppointmentNoticeModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (ScanAppointmentNoticeModel)EmptyFilter.Handler(model);
                model.LocationId = Convert.ToInt32(header.LocationId);
                var foundDeletingItem = await this.unitOfWork.ScanAppointmentNotices.FindAsync(m => m.ScanAppointmentNoticeId == model.ScanAppointmentNoticeId);
                var response = await this.scanAppointmentNoticeService.DeleteAsync(model.ScanAppointmentNoticeId);                
                var machineName = await this.unitOfWork.ScanMachineMasters.FindAsync(p => p.ScanMachineMasterId == foundDeletingItem.MachineId);
                var locationName = await this.unitOfWork.Locations.FindAsync(l => l.LocationId == foundDeletingItem.LocationId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var scanLogModel = new ScanLogModel
                {
                    AccountId = model.ModifiedBy,
                    ScanLogTypeId = (int)ScanLogTypes.ScanAppointmentNotice,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" { model.ModifiedByName } has Deleted Scan Appointment Notice - {foundDeletingItem.Description} for the Scan Machine : <b>{machineName.MachineName}</b> Successfully.",
                    LocationId = model.LocationId
                };
                await this.scanLogServices.LogAsync(scanLogModel);

                return this.Success(" Scan Appointment Notice has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict(" Scan Appointment Notice can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }


        /// <summary>
        /// The delete  Scan Appointment Notice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 -  Scan Appointment Notice deleted successfully.
        /// - 409 -  Scan Appointment Notice can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] ScanAppointmentNoticeModel model, [FromHeader] LocationHeader header)
        {

            model = (ScanAppointmentNoticeModel)EmptyFilter.Handler(model);

            var response = await this.scanAppointmentNoticeService.ModifyStatusAsync(model); 
            var foundDeletingItem = await this.unitOfWork.ScanAppointmentNotices.FindAsync(m => m.ScanAppointmentNoticeId == model.ScanAppointmentNoticeId);
            var scanName = await this.unitOfWork.ScanMachineMasters.FindAsync(p => p.ScanMachineMasterId == model.MachineId);
            var locationName = await this.unitOfWork.Locations.FindAsync(l => l.LocationId == model.LocationId);
            if (response == 0)
            {
                return this.ServerError();
            }

            var scanLogModel = new ScanLogModel
            {
                AccountId = model.CreatedBy,
                ScanLogTypeId = (int)ScanLogTypes.ScanAppointmentNotice,
                LogDate = DateTime.Now,
                LogFrom = (short)model.LoginRoleId,
                LogDescription = $"{model.CreatedByName} has Modified Scan Appointment Notice - {foundDeletingItem.Description} for the Scan Machine : <b>{scanName.MachineName}</b> Successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.scanLogServices.LogAsync(scanLogModel);

            return this.Success("Scan Appointment Notice has been updated the status.");
        }

        [AllowAnonymous]
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> Insert([FromBody] ScanAppointmentNoticeModel model, [FromHeader] LocationHeader header)
        {
            model = (ScanAppointmentNoticeModel)EmptyFilter.Handler(model);
            try
            {
                var response = await this.scanAppointmentNoticeService.InsertAsync(model);
                switch (response)
                {
                    case -1:
                        return this.Conflict(" Scan Appointment Notice has already been exists with us in between dates.");
                    case 0:
                        return this.ServerError();
                }
                var scanName = await this.unitOfWork.ScanMachineMasters.FindAsync(p => p.ScanMachineMasterId == model.MachineId);
                var locationName = await this.unitOfWork.Locations.FindAsync(l => l.LocationId == model.LocationId);
                var scanLogModel = new ScanLogModel
                {
                    AccountId = model.ModifiedBy,
                    ScanLogTypeId = (int)ScanLogTypes.ScanAppointmentNotice,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" { model.ModifiedByName } has Added Scan Appointment Notice - <b>{model.Description}</b> for the Scan Machine : <b>{scanName.MachineName}</b> Successfully.",
                    LocationId = Convert.ToInt32(header.LocationId),
                };
                await this.scanLogServices.LogAsync(scanLogModel);
                return this.Success(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception e)
            {
                return this.Conflict(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = e.Message
                });
            }
        }
        /// <summary>
        /// The update charge.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - charge updated successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("edit")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> EditAsync([FromBody] ScanAppointmentNoticeModel model)
        {
            try
            {
                model = (ScanAppointmentNoticeModel)EmptyFilter.Handler(model);
                var response = await this.scanAppointmentNoticeService.UpdateAsync(model);
                var foundDeletingItem = await this.unitOfWork.ScanAppointmentNotices.FindAsync(m => m.ScanAppointmentNoticeId == model.ScanAppointmentNoticeId);
                var scanName = await this.unitOfWork.ScanMachineMasters.FindAsync(p => p.ScanMachineMasterId == model.MachineId);
                var locationName = await this.unitOfWork.Locations.FindAsync(l => l.LocationId == model.LocationId);
                switch (response)
                {
                    case -1:
                        return this.Conflict(" Scan Appointment Notice is duplicating the existing dates.");
                    case 0:
                        return this.ServerError();
                }
                var scanLogModel = new ScanLogModel
                {
                    AccountId = model.ModifiedBy,
                    ScanLogTypeId = (int)ScanLogTypes.ScanAppointmentNotice,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" { model.ModifiedByName } has Updated Scan Appointment Notice - <b>{model.Description}</b> for the Scan Machine : <b>{scanName.MachineName}</b> Successfully.",
                    LocationId = model.LocationId
                };
                await this.scanLogServices.LogAsync(scanLogModel);
                return this.Success(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception e)
            {
                return this.Conflict(e.Message);
            }
        }
    }
}
